<?php
/**
 * @package   OSMap-Pro
 * @contact   www.joomlashack.com, help@joomlashack.com
 * @copyright 2007-2014 XMap - Joomla! Vargas - Guillermo Vargas. All rights reserved.
 * @copyright 2016-2021 Joomlashack.com. All rights reserved
 * @license   https://www.gnu.org/licenses/gpl.html GNU/GPL
 *
 * This file is part of OSMap-Pro.
 *
 * OSMap-Pro is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * OSMap-Pro is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSMap-Pro.  If not, see <https://www.gnu.org/licenses/>.
 */

use Alledia\OSMap\Helper\General;
use Alledia\OSMap\Sitemap\Collector;
use Alledia\OSMap\Sitemap\Item;
use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use Joomla\Registry\Registry;
use Joomla\Utilities\ArrayHelper;

defined('_JEXEC') or die('Restricted access');

class osmap_com_weblinks
{
    /**
     * @var string[]
     */
    protected static $views = ['categories', 'category'];

    /**
     * @var bool
     */
    protected static $enabled = null;

    /**
     * @var self
     */
    protected static $instance = null;

    public function __construct()
    {
        if (static::isEnabled()) {
            JLoader::register('WeblinksHelperRoute', JPATH_SITE . '/components/com_weblinks/helpers/route.php');
        }
    }

    /**
     * @return self
     */
    public static function getInstance()
    {
        if (empty(static::$instance)) {
            $instance = new self();

            static::$instance = $instance;
        }

        return static::$instance;
    }

    /**
     * @param Collector $osmap
     * @param Item      $parent
     * @param Registry  $params
     *
     * @return void
     * @throws Exception
     */
    public static function getTree($osmap, $parent, $params)
    {
        $uri = new Uri($parent->link);

        if (!static::isEnabled() || !in_array($uri->getVar('view'), static::$views)) {
            return;
        }

        $params['groups']              = General::getAuthorisedViewLevels();
        $params['language_filter']     = Factory::getApplication()->getLanguageFilter();
        $params['include_links']       = ArrayHelper::getValue($params, 'include_links', 1);
        $params['include_links']       = ($params['include_links'] == 1
            || ($params['include_links'] == 2 && $osmap->view == 'xml')
            || ($params['include_links'] == 3 && $osmap->view == 'html')
        );
        $params['show_unauth']         = ArrayHelper::getValue($params, 'show_unauth', 0);
        $params['show_unauth']         = ($params['show_unauth'] == 1
            || ($params['show_unauth'] == 2 && $osmap->view == 'xml')
            || ($params['show_unauth'] == 3 && $osmap->view == 'html')
        );
        $params['category_priority']   = ArrayHelper::getValue($params, 'category_priority', $parent->priority);
        $params['category_changefreq'] = ArrayHelper::getValue($params, 'category_changefreq', $parent->changefreq);

        if ($params['category_priority'] == -1) {
            $params['category_priority'] = $parent->priority;
        }

        if ($params['category_changefreq'] == -1) {
            $params['category_changefreq'] = $parent->changefreq;
        }

        $params['link_priority']   = ArrayHelper::getValue($params, 'link_priority', $parent->priority);
        $params['link_changefreq'] = ArrayHelper::getValue($params, 'link_changefreq', $parent->changefreq);

        if ($params['link_priority'] == -1) {
            $params['link_priority'] = $parent->priority;
        }

        if ($params['link_changefreq'] == -1) {
            $params['link_changefreq'] = $parent->changefreq;
        }
        switch ($uri->getVar('view')) {
            case 'categories':
                static::getCategoryTree($osmap, $parent, $params, $uri->getVar('id'));
                break;

            case 'category':
                static::getlinks($osmap, $parent, $params, $uri->getVar('id'));
                break;
        }
    }

    /**
     * @param Collector $osmap
     * @param Item      $parent
     * @param Registry  $params
     * @param int       $parent_id
     *
     * @return void
     */
    protected static function getCategoryTree($osmap, $parent, $params, $parent_id)
    {
        $db = Factory::getDbo();

        $query = $db->getQuery(true)
            ->select([
                'c.id',
                'c.alias',
                'c.title',
                'c.parent_id, 
                params,
                 metadata'
            ])
            ->from('#__categories AS c')
            ->where([
                'c.parent_id = ' . $db->quote($parent_id ? $parent_id : 1),
                'c.extension = ' . $db->quote('com_weblinks'),
                'c.published = 1'
            ])
            ->order('c.lft ASC');

        if (!$params['show_unauth']) {
            $query->where('c.access IN(' . $params['groups'] . ')');
        }

        if ($params['language_filter']) {
            $query->where(
                sprintf(
                    'c.language IN(%s, %s)',
                    $db->quote(JFactory::getLanguage()->getTag()),
                    $db->quote('*')
                )
            );
        }

        $rows = $db->setQuery($query)->loadObjectList();

        if (empty($rows)) {
            return;
        }

        $osmap->changeLevel(1);

        foreach ($rows as $row) {
            $node = (object)[
                'id'         => $parent->id,
                'name'       => $row->title,
                'uid'        => $parent->uid . '_cid_' . $row->id,
                'browserNav' => $parent->browserNav,
                'priority'   => $params['category_priority'],
                'changefreq' => $params['category_changefreq'],
                'pid'        => $row->parent_id,
                'link'       => WeblinksHelperRoute::getCategoryRoute($row->id)
            ];

            if ($osmap->printNode($node) !== false) {
                static::getlinks($osmap, $parent, $params, $row->id);
            }
        }

        $osmap->changeLevel(-1);
    }

    /**
     * @param Collector $osmap
     * @param Item      $parent
     * @param Registry  $params
     * @param int       $catid
     *
     * @return void
     */
    protected static function getlinks($osmap, $parent, $params, $catid)
    {
        static::getCategoryTree($osmap, $parent, $params, $catid);

        if (!$params['include_links']) {
            return;
        }

        $db  = Factory::getDbo();
        $now = Factory::getDate('now', 'UTC')->toSql();

        $query = $db->getQuery(true)
            ->select([
                'w.id',
                'w.alias',
                'w.title',
                'params',
                'metadata'
            ])
            ->from('#__weblinks AS w')
            ->where([
                'w.catid = ' . (int)$catid,
                'w.state = 1',
                sprintf(
                    '(w.publish_up = %s OR w.publish_up <= %s)',
                    $db->quote($db->getNullDate()),
                    $db->quote($now)
                ),
                sprintf(
                    '(w.publish_down = %s OR w.publish_down >= %s)',
                    $db->quote($db->getNullDate()),
                    $db->quote($now)
                )
            ])
            ->order('w.ordering ASC');

        if (!$params['show_unauth']) {
            $query->where(sprintf('w.access IN(%s)', $params['groups']));
        }

        if ($params['language_filter']) {
            $query->where(
                sprintf(
                    'w.language IN(%s, %s)',
                    $db->quote(Factory::getLanguage()->getTag()),
                    $db->quote('*')
                )
            );
        }

        $rows = $db->setQuery($query)->loadObjectList();

        if (empty($rows)) {
            return;
        }

        $osmap->changeLevel(1);

        foreach ($rows as $row) {
            $node = (object)[
                'id'         => $parent->id,
                'name'       => $row->title,
                'uid'        => $parent->uid . '_' . $row->id,
                'browserNav' => $parent->browserNav,
                'priority'   => $params['link_priority'],
                'changefreq' => $params['link_changefreq'],
                'link'       => WeblinksHelperRoute::getWeblinkRoute($row->id . ':' . $row->alias, $catid)
            ];

            $osmap->printNode($node);
        }

        $osmap->changeLevel(-1);
    }

    /**
     * @return bool
     */
    protected static function isEnabled()
    {
        if (null === static::$enabled) {
            $db = Factory::getDbo();
            $db->setQuery('SELECT enabled FROM `#__extensions` WHERE name=' . $db->quote('com_weblinks'));
            static::$enabled = (bool)$db->loadResult();
        }

        return static::$enabled;
    }
}
